/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.plugin.audit;

import com.google.common.collect.Queues;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CodingErrorAction;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.doris.catalog.Env;
import org.apache.doris.common.Config;
import org.apache.doris.plugin.AuditEvent;
import org.apache.doris.plugin.AuditPlugin;
import org.apache.doris.plugin.Plugin;
import org.apache.doris.plugin.PluginContext;
import org.apache.doris.plugin.PluginException;
import org.apache.doris.plugin.PluginInfo;
import org.apache.doris.plugin.audit.DorisStreamLoader;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class AuditLoaderPlugin
extends Plugin
implements AuditPlugin {
    private static final Logger LOG = LogManager.getLogger(AuditLoaderPlugin.class);
    private static final DateTimeFormatter DATETIME_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSS").withZone(ZoneId.systemDefault());
    private StringBuilder auditLogBuffer = new StringBuilder();
    private StringBuilder slowLogBuffer = new StringBuilder();
    private long lastLoadTimeAuditLog = 0L;
    private long lastLoadTimeSlowLog = 0L;
    private BlockingQueue<AuditEvent> auditEventQueue;
    private DorisStreamLoader streamLoader;
    private Thread loadThread;
    private AuditLoaderConf conf;
    private volatile boolean isClosed = false;
    private volatile boolean isInit = false;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void init(PluginInfo info, PluginContext ctx) throws PluginException {
        super.init(info, ctx);
        AuditLoaderPlugin auditLoaderPlugin = this;
        synchronized (auditLoaderPlugin) {
            if (this.isInit) {
                return;
            }
            this.lastLoadTimeAuditLog = System.currentTimeMillis();
            this.lastLoadTimeSlowLog = System.currentTimeMillis();
            this.loadConfig(ctx, info.getProperties());
            this.auditEventQueue = Queues.newLinkedBlockingDeque((int)this.conf.maxQueueSize);
            this.streamLoader = new DorisStreamLoader(this.conf);
            this.loadThread = new Thread((Runnable)new LoadWorker(this.streamLoader), "audit loader thread");
            this.loadThread.start();
            this.isInit = true;
        }
    }

    private void loadConfig(PluginContext ctx, Map<String, String> pluginInfoProperties) throws PluginException {
        Path pluginPath = FileSystems.getDefault().getPath(ctx.getPluginPath(), new String[0]);
        if (!Files.exists(pluginPath, new LinkOption[0])) {
            throw new PluginException("plugin path does not exist: " + pluginPath);
        }
        Path confFile = pluginPath.resolve("plugin.conf");
        if (!Files.exists(confFile, new LinkOption[0])) {
            throw new PluginException("plugin conf file does not exist: " + confFile);
        }
        Properties props = new Properties();
        try (InputStream stream = Files.newInputStream(confFile, new OpenOption[0]);){
            props.load(stream);
        }
        catch (IOException e) {
            throw new PluginException(e.getMessage());
        }
        for (Map.Entry<String, String> entry : pluginInfoProperties.entrySet()) {
            props.setProperty(entry.getKey(), entry.getValue());
        }
        Map<String, String> properties = props.stringPropertyNames().stream().collect(Collectors.toMap(Function.identity(), props::getProperty));
        this.conf = new AuditLoaderConf();
        this.conf.init(properties);
        this.conf.feIdentity = ctx.getFeIdentity();
    }

    public void close() throws IOException {
        super.close();
        this.isClosed = true;
        if (this.loadThread != null) {
            try {
                this.loadThread.join();
            }
            catch (InterruptedException e) {
                LOG.debug("encounter exception when closing the audit loader", (Throwable)e);
            }
        }
    }

    public boolean eventFilter(AuditEvent.EventType type) {
        return type == AuditEvent.EventType.AFTER_QUERY;
    }

    public void exec(AuditEvent event) {
        try {
            this.auditEventQueue.add(event);
        }
        catch (Exception e) {
            LOG.debug("encounter exception when putting current audit batch, discard current audit event", (Throwable)e);
        }
    }

    private void assembleAudit(AuditEvent event) {
        if (this.conf.enableSlowLog && event.queryTime > Config.qe_slow_log_ms) {
            this.fillLogBuffer(event, this.slowLogBuffer);
        }
        this.fillLogBuffer(event, this.auditLogBuffer);
    }

    private void fillLogBuffer(AuditEvent event, StringBuilder logBuffer) {
        logBuffer.append(event.queryId).append("\t");
        logBuffer.append(AuditLoaderPlugin.longToTimeString(event.timestamp)).append("\t");
        logBuffer.append(event.clientIp).append("\t");
        logBuffer.append(event.user).append("\t");
        logBuffer.append(event.db).append("\t");
        logBuffer.append(event.state).append("\t");
        logBuffer.append(event.errorCode).append("\t");
        logBuffer.append(event.errorMessage).append("\t");
        logBuffer.append(event.queryTime).append("\t");
        logBuffer.append(event.scanBytes).append("\t");
        logBuffer.append(event.scanRows).append("\t");
        logBuffer.append(event.returnRows).append("\t");
        logBuffer.append(event.stmtId).append("\t");
        logBuffer.append(event.isQuery ? 1 : 0).append("\t");
        logBuffer.append(event.feIp).append("\t");
        logBuffer.append(event.cpuTimeMs).append("\t");
        logBuffer.append(event.sqlHash).append("\t");
        logBuffer.append(event.sqlDigest).append("\t");
        logBuffer.append(event.peakMemoryBytes).append("\t");
        String stmt = this.truncateByBytes(event.stmt).replace("\n", " ").replace("\t", " ").replace("\r", " ");
        LOG.debug("receive audit event with stmt: {}", (Object)stmt);
        logBuffer.append(stmt).append("\n");
    }

    private String truncateByBytes(String str) {
        int maxLen = Math.min(this.conf.max_stmt_length, str.getBytes().length);
        if (maxLen >= str.getBytes().length) {
            return str;
        }
        Charset utf8Charset = Charset.forName("UTF-8");
        CharsetDecoder decoder = utf8Charset.newDecoder();
        byte[] sb = str.getBytes();
        ByteBuffer buffer = ByteBuffer.wrap(sb, 0, maxLen);
        CharBuffer charBuffer = CharBuffer.allocate(maxLen);
        decoder.onMalformedInput(CodingErrorAction.IGNORE);
        decoder.decode(buffer, charBuffer, true);
        decoder.flush(charBuffer);
        return new String(charBuffer.array(), 0, charBuffer.position());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void loadIfNecessary(DorisStreamLoader loader, boolean slowLog) {
        StringBuilder logBuffer = slowLog ? this.slowLogBuffer : this.auditLogBuffer;
        long lastLoadTime = slowLog ? this.lastLoadTimeSlowLog : this.lastLoadTimeAuditLog;
        long currentTime = System.currentTimeMillis();
        if ((long)logBuffer.length() >= this.conf.maxBatchSize || currentTime - lastLoadTime >= this.conf.maxBatchIntervalSec * 1000L) {
            try {
                String token = "";
                if (this.conf.use_auth_token) {
                    try {
                        token = Env.getCurrentEnv().getLoadManager().getTokenManager().acquireToken();
                    }
                    catch (Exception e) {
                        LOG.error("Failed to get auth token: {}", (Throwable)e);
                    }
                }
                DorisStreamLoader.LoadResponse response = loader.loadBatch(logBuffer, slowLog, token);
                LOG.debug("audit loader response: {}", (Object)response);
            }
            catch (Exception e) {
                LOG.debug("encounter exception when putting current audit batch, discard current batch", (Throwable)e);
            }
            finally {
                this.resetLogBufferAndLastLoadTime(currentTime, slowLog);
            }
        }
    }

    private void resetLogBufferAndLastLoadTime(long currentTime, boolean slowLog) {
        if (slowLog) {
            this.slowLogBuffer = new StringBuilder();
            this.lastLoadTimeSlowLog = currentTime;
        } else {
            this.auditLogBuffer = new StringBuilder();
            this.lastLoadTimeAuditLog = currentTime;
        }
    }

    public static String longToTimeString(long timeStamp) {
        if (timeStamp <= 0L) {
            return "1900-01-01 00:00:00.000";
        }
        return DATETIME_FORMAT.format(LocalDateTime.ofInstant(Instant.ofEpochMilli(timeStamp), ZoneId.systemDefault()));
    }

    private class LoadWorker
    implements Runnable {
        private DorisStreamLoader loader;

        public LoadWorker(DorisStreamLoader loader) {
            this.loader = loader;
        }

        @Override
        public void run() {
            while (!AuditLoaderPlugin.this.isClosed) {
                try {
                    AuditEvent event = (AuditEvent)AuditLoaderPlugin.this.auditEventQueue.poll(5L, TimeUnit.SECONDS);
                    if (event == null) continue;
                    AuditLoaderPlugin.this.assembleAudit(event);
                    if (((AuditLoaderPlugin)AuditLoaderPlugin.this).conf.enableSlowLog) {
                        AuditLoaderPlugin.this.loadIfNecessary(this.loader, true);
                    }
                    AuditLoaderPlugin.this.loadIfNecessary(this.loader, false);
                }
                catch (InterruptedException ie) {
                    LOG.debug("encounter exception when loading current audit batch", (Throwable)ie);
                }
                catch (Exception e) {
                    LOG.error("run audit logger error:", (Throwable)e);
                }
            }
        }
    }

    public static class AuditLoaderConf {
        public static final String PROP_MAX_BATCH_SIZE = "max_batch_size";
        public static final String PROP_MAX_BATCH_INTERVAL_SEC = "max_batch_interval_sec";
        public static final String PROP_MAX_QUEUE_SIZE = "max_queue_size";
        public static final String PROP_FRONTEND_HOST_PORT = "frontend_host_port";
        public static final String PROP_USER = "user";
        public static final String PROP_PASSWORD = "password";
        public static final String PROP_DATABASE = "database";
        public static final String PROP_TABLE = "table";
        public static final String PROP_AUDIT_LOG_TABLE = "audit_log_table";
        public static final String PROP_SLOW_LOG_TABLE = "slow_log_table";
        public static final String PROP_ENABLE_SLOW_LOG = "enable_slow_log";
        public static final String MAX_STMT_LENGTH = "max_stmt_length";
        public static final String USE_AUTH_TOKEN = "use_auth_token";
        public long maxBatchSize = 0x3200000L;
        public long maxBatchIntervalSec = 60L;
        public int maxQueueSize = 1000;
        public String frontendHostPort = "127.0.0.1:8030";
        public String user = "root";
        public String password = "";
        public String database = "doris_audit_db__";
        public String auditLogTable = "doris_audit_log_tbl__";
        public String slowLogTable = "doris_slow_log_tbl__";
        public boolean enableSlowLog = false;
        public String feIdentity = "";
        public int max_stmt_length = 4096;
        public boolean use_auth_token = false;

        public void init(Map<String, String> properties) throws PluginException {
            try {
                if (properties.containsKey(PROP_MAX_BATCH_SIZE)) {
                    this.maxBatchSize = Long.valueOf(properties.get(PROP_MAX_BATCH_SIZE));
                }
                if (properties.containsKey(PROP_MAX_BATCH_INTERVAL_SEC)) {
                    this.maxBatchIntervalSec = Long.valueOf(properties.get(PROP_MAX_BATCH_INTERVAL_SEC));
                }
                if (properties.containsKey(PROP_MAX_QUEUE_SIZE)) {
                    this.maxQueueSize = Integer.valueOf(properties.get(PROP_MAX_QUEUE_SIZE));
                }
                if (properties.containsKey(PROP_FRONTEND_HOST_PORT)) {
                    this.frontendHostPort = properties.get(PROP_FRONTEND_HOST_PORT);
                }
                if (properties.containsKey(PROP_USER)) {
                    this.user = properties.get(PROP_USER);
                }
                if (properties.containsKey(PROP_PASSWORD)) {
                    this.password = properties.get(PROP_PASSWORD);
                }
                if (properties.containsKey(PROP_DATABASE)) {
                    this.database = properties.get(PROP_DATABASE);
                }
                if (properties.containsKey(PROP_TABLE)) {
                    this.auditLogTable = properties.get(PROP_TABLE);
                }
                if (properties.containsKey(PROP_AUDIT_LOG_TABLE)) {
                    this.auditLogTable = properties.get(PROP_AUDIT_LOG_TABLE);
                }
                if (properties.containsKey(PROP_SLOW_LOG_TABLE)) {
                    this.slowLogTable = properties.get(PROP_SLOW_LOG_TABLE);
                }
                if (properties.containsKey(PROP_ENABLE_SLOW_LOG)) {
                    this.enableSlowLog = Boolean.valueOf(properties.get(PROP_ENABLE_SLOW_LOG));
                }
                if (properties.containsKey(MAX_STMT_LENGTH)) {
                    this.max_stmt_length = Integer.parseInt(properties.get(MAX_STMT_LENGTH));
                }
                if (properties.containsKey(USE_AUTH_TOKEN)) {
                    this.use_auth_token = Boolean.valueOf(properties.get(USE_AUTH_TOKEN));
                }
            }
            catch (Exception e) {
                throw new PluginException(e.getMessage());
            }
        }
    }
}

